import { useRouter } from "next/router";
import { useEffect, useMemo, useState } from "react";
import { api } from "../../src/lib/api";
import { MapPicker } from "../../src/components/MapPicker";
import Link from "next/link";

type Product = { id:number; name:string; price:number };
type CartItem = { product_id:number; name:string; price:number; qty:number };

export default function VendorProducts() {
  const router = useRouter();
  const vendorId = Number(router.query.id);
  const [products, setProducts] = useState<Product[]>([]);
  const [cart, setCart] = useState<CartItem[]>([]);
  const [drop, setDrop] = useState<{lat:number; lng:number} | null>(null);
  const [quote, setQuote] = useState<any>(null);
  const [msg, setMsg] = useState<string>("");

  useEffect(() => {
    if (!vendorId) return;
    api<Product[]>(`/api/vendors/${vendorId}/products`).then(setProducts).catch(e=>setMsg(e.message));
  }, [vendorId]);

  const subtotal = useMemo(() => cart.reduce((s,i)=>s + i.price*i.qty, 0), [cart]);

  const add = (p:Product) => {
    setCart(prev => {
      const ex = prev.find(x=>x.product_id===p.id);
      if (ex) return prev.map(x=>x.product_id===p.id ? {...x, qty:x.qty+1} : x);
      return [...prev, { product_id:p.id, name:p.name, price:p.price, qty:1 }];
    });
  };

  const getQuote = async () => {
    setMsg("");
    if (!drop) return setMsg("Pick location on map first.");
    const q = await api<any>("/api/orders/quote", {
      method:"POST",
      body: JSON.stringify({ vendor_id: vendorId, dropoff_lat: drop.lat, dropoff_lng: drop.lng, type:"delivery" })
    });
    setQuote(q);
  };

  const checkout = async () => {
    setMsg("");
    if (!drop) return setMsg("Pick location first.");
    if (!quote) return setMsg("Get quote first.");

    const payload = {
      vendor_id: vendorId,
      items: cart.map(i => ({ product_id:i.product_id, qty:i.qty })),
      dropoff_address: "Jeddah",
      dropoff_lat: drop.lat,
      dropoff_lng: drop.lng,
      type: "delivery",
      is_scheduled: false,
    };
    const res = await api<any>("/api/orders", { method:"POST", body: JSON.stringify(payload) });
    setCart([]);
    setMsg(`Order created ✅ #${res.id}`);
    router.push(`/order/${res.id}`);
  };

  return (
    <div style={{ maxWidth: 820, margin:"24px auto", padding:16 }}>
      <div style={{ display:"flex", justifyContent:"space-between" }}>
        <h2>Products</h2>
        <Link href="/">Back</Link>
      </div>

      {msg && <p>{msg}</p>}

      {products.map(p => (
        <div key={p.id} style={{ border:"1px solid #ddd", padding:12, marginTop:10, borderRadius:10 }}>
          <div><b>{p.name}</b> — {p.price} SAR</div>
          <button onClick={()=>add(p)} style={{ marginTop:8 }}>Add</button>
        </div>
      ))}

      <hr style={{ margin:"20px 0" }} />
      <h3>Drop-off location</h3>
      <MapPicker onPick={(lat,lng)=>{ setDrop({lat,lng}); setQuote(null); }} />
      <button onClick={getQuote} style={{ marginTop:10, padding:10 }}>احسب سعر التوصيل</button>

      {quote && (
        <div style={{ marginTop:10, padding:12, border:"1px solid #ddd", borderRadius:10 }}>
          <div>الفرع: <b>{quote.branch.name}</b></div>
          <div>المسافة: <b>{quote.distance_km} km</b></div>
          <div>رسوم التوصيل: <b>{quote.delivery_fee} SAR</b></div>
        </div>
      )}

      <hr style={{ margin:"20px 0" }} />
      <h3>Cart</h3>
      {cart.map(i => <div key={i.product_id}>{i.name} × {i.qty}</div>)}
      <div style={{ marginTop: 8 }}><b>Subtotal:</b> {subtotal} SAR</div>
      <button onClick={checkout} style={{ marginTop: 10, padding: 10 }}>Checkout</button>
    </div>
  );
}
